<?php
/**
 * Plugin Name: VS Security
 * Description: A security plugin to block new user registrations, prevent search console verification file uploads, and add additional layers of security.
 * Version: 1.0
 * Author: Zargham
 */

// Disable new user registrations (both admin and author)
function vs_security_disable_user_registration() {
    // Disable user registration globally if it is enabled
    if ( get_option( 'users_can_register' ) ) {
        update_option( 'users_can_register', 0 );
    }

    // Block admin registration
    if ( is_admin() && isset($_GET['action']) && $_GET['action'] == 'adduser') {
        wp_die( 'User registration is disabled.' );
    }
}
add_action( 'init', 'vs_security_disable_user_registration' );

// Use plugin activation to ensure registration is disabled
function vs_security_on_activation() {
    update_option( 'users_can_register', 0 );
}
register_activation_hook( __FILE__, 'vs_security_on_activation' );

// Prevent uploading search console verification files
function vs_security_block_search_console_files( $file ) {
    $blocked_files = ['google', 'bing', 'yandex']; // List of search engines' verification keywords
    foreach ($blocked_files as $blocked_file) {
        if ( strpos( strtolower($file['name']), $blocked_file ) !== false ) {
            $file['error'] = 'Uploading search console verification files is not allowed.';
        }
    }
    return $file;
}
add_filter( 'wp_handle_upload_prefilter', 'vs_security_block_search_console_files' );

// Disable XML-RPC (a common attack vector)
add_filter( 'xmlrpc_enabled', '__return_false' );

// Prevent theme and plugin file editing from the dashboard
define('DISALLOW_FILE_EDIT', true);


// Hide WordPress version (prevents attackers from knowing the version of WordPress)
remove_action('wp_head', 'wp_generator');

// Disable WordPress REST API for non-authenticated users (prevents unauthorized access)
function vs_security_disable_rest_api() {
    if ( !is_user_logged_in() ) {
        wp_die( 'REST API access is disabled.' );
    }
}
add_action( 'rest_api_init', 'vs_security_disable_rest_api', 1 );

// Disable directory listing
function vs_security_disable_directory_listing() {
    if ( is_admin() ) {
        return;
    }

    // .htaccess directive to disable directory listing
    if ( file_exists( ABSPATH . '.htaccess' ) ) {
        $htaccess = file_get_contents( ABSPATH . '.htaccess' );
        if ( strpos( $htaccess, 'Options -Indexes' ) === false ) {
            file_put_contents( ABSPATH . '.htaccess', "\nOptions -Indexes\n", FILE_APPEND );
        }
    }
}
add_action( 'init', 'vs_security_disable_directory_listing' );

// Security headers to prevent some common vulnerabilities
function vs_security_add_headers() {
    header( 'X-Content-Type-Options: nosniff' );
    header( 'X-Frame-Options: DENY' );
    header( 'Strict-Transport-Security: max-age=31536000; includeSubDomains; preload' );
    header( 'Referrer-Policy: no-referrer-when-downgrade' );
    header( 'Permissions-Policy: geolocation=(), microphone=(), camera=()' );
}
add_action( 'send_headers', 'vs_security_add_headers' );

// Additional security: Block access to wp-config.php
function vs_security_block_wp_config() {
    if ( strpos( $_SERVER['REQUEST_URI'], 'wp-config.php' ) !== false ) {
        wp_die( 'Access to wp-config.php is forbidden.' );
    }
}
add_action( 'init', 'vs_security_block_wp_config' );

// Block admin user creation (even from the "Add New User" page)
function vs_security_block_admin_user_creation() {
    if ( is_admin() && isset( $_GET['action'] ) && $_GET['action'] === 'adduser' ) {
        wp_die( 'User registration is disabled for this site.' );
    }
}
add_action( 'admin_init', 'vs_security_block_admin_user_creation' );

// Block the capability to create users for administrators
function vs_security_block_create_user_capability($allcaps) {
    if (isset($allcaps['create_users'])) {
        $allcaps['create_users'] = false;
    }
    return $allcaps;
}
add_filter('user_has_cap', 'vs_security_block_create_user_capability');
